/**
 * server.c
 *
 * Програма моделює роботу послідовного сервера, який отримує
 * клієнтські запити через датаграмний сокет в комунікаційному домені
 * AF_UNIX. Отримавши черговий клієнтський запит, сервер просто повертає
 * його назад клієнту (працює як сервер служби "Луна"). Сервер 
 * прив'язується до сокета з ім'ям, заданим константою SERVER_ADDRESS
 *
 */

#include "myecho.h"

#include <netdb.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/un.h>

/*
 * Журнал помилок
 */
#define ERRLOGFILE "server.err"

/*
 * Обробник сигналу SIGTERM
 */
void sigterm_handler()
{
	unlink(SERVER_ADDRESS);
	exit(EXIT_SUCCESS);
}

int main()
{
	int s;						// Дескриптор серверного сокета
	struct sockaddr_un serv_addr;	// Адреса сервера
	socklen_t saddrlen;			// Довжина адреси сервера
	struct sigaction act;
    
	// Приєднує стандартний потік виведення повідомлень про помилки
	// до файлу ERRLOGFILE
	if ((freopen(ERRLOGFILE, "w", stderr)) == NULL)
		exit(EXIT_FAILURE);

	// Реєструє обробник сигналу SIGTERM
	memset(&act, 0, sizeof(act));
	act.sa_handler = sigterm_handler;	
	if (sigaction(SIGTERM, &act, NULL) == -1) {
		perror("Error setting SIGTERM handler");
		exit(EXIT_FAILURE);
	}

	// Створює датаграмний сокет в домені AF_UNIX
	s = socket(AF_LOCAL, SOCK_DGRAM, 0);
	if (s == -1) {
		perror("Error creating server socket");
		exit(EXIT_FAILURE);
	}

	// Розміщує в структурі serv_addr свою адресу, а в змінній saddrlen - її довжину
	memset(&serv_addr, 0, sizeof(serv_addr));
	serv_addr.sun_family = AF_UNIX;
	strcpy(serv_addr.sun_path, SERVER_ADDRESS);
	saddrlen = sizeof(serv_addr.sun_family) + strlen(serv_addr.sun_path);
	// Прив'язує сокет s до локальної адреси
	unlink(SERVER_ADDRESS);
	if (bind(s, (struct sockaddr *) &serv_addr, saddrlen) == -1) {
		perror("Error binding server socket");
		exit(EXIT_FAILURE);
	}

	while (1) {
		struct sockaddr_un clnt_addr;	// Адреса клієнта
		socklen_t caddrlen;		// Довжина адреси клієнта
		char buf[BUFSIZE];		// Буфер для повідомлень
		ssize_t n_recved;

		// Очищає структуру для клієнтської адреси
		memset(&clnt_addr, 0, sizeof(clnt_addr));
		caddrlen = sizeof(clnt_addr);

		// Отримує датаграму
		n_recved = recvfrom(s, buf, sizeof(buf), 0,
		                    (struct sockaddr *) &clnt_addr, &caddrlen);
		if (n_recved > 0) {
			// Відправляє датаграму назад
			if (sendto(s, buf, n_recved, 0,
			           (struct sockaddr *) &clnt_addr, caddrlen) == -1)
				perror("Error sending datagram to client");
		} else if (n_recved == -1)
			perror("Error receiving datagram from client");
	}

	unlink(SERVER_ADDRESS);
	exit(EXIT_SUCCESS);
}
